<?php

/**
 * Class altDirectoryUpdater
 * 
 * To be included with any plugin or theme to check for and install updates.
 * 
 * Usage: $altDirectoryPackageUpdater = new altDirectoryPackageUpdater( __FILE__ ); in the plugin's main file.
 */
class altDirectoryPackageUpdater {

    protected $package_data;            // Data about the current version of the package
    protected $package_file;            // The directory name and file name of the current package (they should match)
    protected $package_slug;            // The directory name of the package
    protected $package_version;         // The currently installed version number
    protected $package_file_path;       // The full file path to the currently installed package
    protected $package_update_info;     // Update data from the Alt Directory API

    public function __construct( $package_file_path ) {

        add_filter( 'pre_set_site_transient_update_plugins', array( $this, 'setPackageUpdateTransient' ) );
        add_filter( 'plugins_api', array( $this, 'getPackageUpdateInfo' ), 20, 3 );

        $this->package_file_path    = $package_file_path;

        $this->setPackageData();
    }

    /**
     * Sets up the classes properties
     */
    public function setPackageData() {
        if ( ! file_exists( $this->package_file_path ) ) wp_die('Alt Directory updater package file path not found.' );

        if ( ! function_exists('get_plugin_data') ) {
            require_once( ABSPATH . 'wp-admin/includes/plugin.php' );
        }

        $this->package_data     = get_plugin_data( $this->package_file_path );
        $this->package_file     = plugin_basename( $this->package_file_path );
        $this->package_slug     = dirname( plugin_basename( $this->package_file_path ) );
        $this->package_version  = $this->package_data['Version'];
    }

    /**
     * Adds any available updates to the WP update transient
     */
    public function setPackageUpdateTransient( $transient ) {
        
        if ( empty( $transient->checked ) ) {
            return $transient;
        }
    
        $package_update_available = $this->checkPackageUpdate();
    
        if (
            $package_update_available
            && version_compare( $this->package_version, $package_update_available->version, '<' )
            && version_compare( $package_update_available->requires, get_bloginfo( 'version' ), '<=' )
            && version_compare( $package_update_available->requires_php, phpversion(), '<=' )
            && $this->package_slug === $package_update_available->slug
        ) {
            $updated_package                = new stdClass();
            $updated_package->slug          = $package_update_available->slug;
            $updated_package->plugin        = $this->package_file;
            $updated_package->new_version   = $package_update_available->version;
            $updated_package->tested        = $package_update_available->tested;
            $updated_package->package       = $package_update_available->package;
            $updated_package->icons         = array(
                '1x' => $package_update_available->icons->low,
                '2x' => $package_update_available->icons->high,
            );
    
            $transient->response[ $updated_package->plugin ] = $updated_package;
        }
        return $transient;
    }

    /**
     * Retrieves the update info for the Plugin screen
     */
    public function getPackageUpdateInfo( $result, $action, $args ) {
        if ( 'plugin_information' !== $action ) {
            return $result;
        }
    
        if ( $this->package_slug !== $args->slug ) {
            return $result;
        }

		$package_info = $this->checkPackageUpdate();

		if ( ! $package_info ) {
			return $result;
		}

		$result                 = new stdClass();
		$result->name           = $package_info->name;
		$result->slug           = $package_info->slug;
		$result->author         = $package_info->author;
		$result->author_profile = $package_info->author_profile;
		$result->version        = $package_info->version;
		$result->tested         = $package_info->tested;
		$result->requires       = $package_info->requires;
		$result->requires_php   = $package_info->requires_php;
		$result->download_link  = $package_info->package;
		$result->trunk          = $package_info->package;
		$result->last_updated   = $package_info->last_updated;
		$result->sections       = array(
			'description'  => $package_info->sections->description,
			'changelog'    => $package_info->sections->changelog
		);
		$result->icons           = array(
			'1x' => $package_info->icons->low,
			'2x' => $package_info->icons->high,
		);
		if ( ! empty( $package_info->sections->screenshots ) ) {
			$result->sections[ 'screenshots' ] = $package_info->sections->screenshots;
		}

		$result->banners = array(
			'low'  => $package_info->banners->low,
			'high' => $package_info->banners->high
		);

		return $result;
    }

    /**
     * Checks the Alt Directory API for updates to the current package
     * (Please note, it calls the domain name of the Author's URI in the package)
     */
    public function checkPackageUpdate() {
        $alt_data           = wp_remote_get( 
                                $this->package_data['AuthorURI'] . '/alt/packages.json',
                                array(
                                    'timeout' => 10,
                                    'headers' => array(
						                'Accept' => 'application/json'
					                )
                                )
                            );
        
        if ( is_wp_error( $alt_data ) ) {
            return false;
        }

        $alt_packages       = json_decode( wp_remote_retrieve_body( $alt_data ) );

        $update_data        = alt_filter_package_data( $alt_packages, $this->package_slug );

        // If $update_data is null, it means the packages payload doesn't contain information for the current package
        if ( $update_data === NULL ) {
            return false;
        }

        if ( $this->package_version <= $update_data->version ) {
            return $update_data;
        } else {
            return false;
        }
    }

}

function alt_filter_package_data($array, $searchValue) { // May not be needed if I rethink the API response
    if ( is_array( $array ) ) {
        foreach ($array as $object) {
            if ( property_exists($object, 'slug') && $object->slug === $searchValue ) {
                return $object;
            }
        }
    } else {
        // Array is really a single object
        if ( is_object( $array ) ) {
            if ( property_exists($array, 'slug') && $array->slug === $searchValue ) {
                return $array;
            }
        }
    }
    return null; // Return null if the value is not found
}

add_filter( 'https_ssl_verify', '__return_false' ); // I had to use this for Local, I think